----------------
title : C++ 레퍼런스 - any_of 함수
cat_title : any_of
ref_title : any_of, std::any_of
path : /C++ Reference/algorithm
publish_date : 2019-01-17
----------------

##@ cpp-ref-start

#@ any_of

`<algorithm>` 에 정의됨

```cpp-formatted
template <class InputIterator, class UnaryPredicate>
bool any_of(InputIterator first, InputIterator last, UnaryPredicate pred);
```

만일 범위 내의 (`first` 부터 `last` 전 까지) 원소들에 대해 `pred` 가 참을 리턴하는 원소가 하나라도 있다면, `any_of` 역시 참을 리턴합니다. 만일 해당 원소들에 대해 `pred` 가 모두 거짓을 리턴한다면 `any_of` 도 거짓을 리턴합니다.

만일 `[first, last)` 가 비어 있다면 (즉 `first == last` 일 경우) `any_of` 는 `false` 를 리턴합니다.

### 인자들

* `first, last` : 원소들의 시작점과 끝점을 가리키는 반복자(`InputIterator`)들. 이 때, 확인하는 범위는 `[first, last)` 로 정의됩니다. `first` 가 가리키는 원소는 포함되지만 `last` 가 가리키는 원소는 포함되지 않습니다.

* `pred` : 인자를 한 개만 가지는 함수로 `bool` 로 변환 가능한 타입을 리턴해야 합니다.

### 리턴값

만일 원소 한 개라도 `pred` 가 `true` 를 리턴했다면 `true` 를, 아니면 `false` 를 리턴합니다.

### 구현 예시

```cpp-formatted
template <class InputIterator, class UnaryPredicate>
bool any_of(InputIterator first, InputIterator last, UnaryPredicate pred) {
  while (first != last) {
    if (pred(*first)) return true;
    ++first;
  }
  return false;
}
```

아래는 `std::find_if` 를 사용한 방식입니다. 참고로 `std::find_if` 는 조건을 만족하는 첫 번째 원소를 반환합니다.

```cpp-formatted
template <class InputIt, class UnaryPredicate>
constexpr bool any_of(InputIt first, InputIt last, UnaryPredicate p) {
  return std::find_if(first, last, p) != last;
}
```

### 실행 예제

```cpp-formatted
#include <algorithm>  // std::any_of
#include <array>      // std::array
#include <iostream>   // std::cout

int main() {
  std::array<int, 7> foo = {0, 1, -1, 3, -3, 5, -5};

  if (std::any_of(foo.begin(), foo.end(), [](int i) { return i < 0; }))
    std::cout << "범위 내에 음수가 있습니다.\n";

  return 0;
}
```

실행 결과

```exec
범위 내에 음수가 있습니다.
```

### 참고 자료

* `all_of` : 범위 안에 모든 원소들이 조건을 만족하는지 확인한다.
* `none_of` : 범위 안에 모든 원소들이 조건을 만족하지 않는지 확인한다.
* `find_if` : 범위 안에 원소들 중 조건과 일치하는 원소를 찾는다.